"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu"
import { User, LogOut, Settings, UserCog } from "lucide-react"
import Link from "next/link"
import { useRouter } from "next/navigation"

export function AdminUserMenu() {
  const router = useRouter()
  const [adminEmail, setAdminEmail] = useState("admin@teertham.com")

  // Read email from cookie on mount (client-side only)
  useEffect(() => {
    try {
      const cookies = document.cookie.split(";")
      const sessionCookie = cookies.find((c) => c.trim().startsWith("admin_session="))
      if (sessionCookie) {
        const value = sessionCookie.split("=")[1]
        const parsed = JSON.parse(decodeURIComponent(value))
        if (parsed.email) {
          setAdminEmail(parsed.email)
        }
      }
    } catch {
      // Keep default email if parsing fails
    }
  }, [])

  const handleLogout = () => {
    if (typeof window !== "undefined") {
      sessionStorage.removeItem("admin_session")
      document.cookie = "admin_session=; path=/; max-age=0; SameSite=Strict"
    }
    router.push("/")
  }

  return (
    <DropdownMenu>
      <DropdownMenuTrigger asChild>
        <Button
          variant="ghost"
          size="icon"
          className="rounded-xl ring-2 ring-border hover:ring-primary/30 transition-all h-10 w-10"
          aria-label="Open user menu"
        >
          <div className="h-8 w-8 rounded-lg bg-gradient-to-br from-primary/20 to-primary/5 flex items-center justify-center">
            <User className="h-4 w-4 text-primary" aria-hidden="true" />
          </div>
        </Button>
      </DropdownMenuTrigger>
      <DropdownMenuContent align="end" className="w-60 p-2">
        <DropdownMenuLabel className="font-normal p-3 bg-muted/50 rounded-lg mb-2">
          <div className="flex flex-col space-y-1">
            <p className="text-sm font-semibold leading-none">Admin Account</p>
            <p className="text-xs leading-none text-muted-foreground">{adminEmail}</p>
          </div>
        </DropdownMenuLabel>
        <DropdownMenuItem asChild className="rounded-lg py-2.5 cursor-pointer">
          <Link href="/admin/settings">
            <Settings className="mr-3 h-4 w-4" aria-hidden="true" />
            Admin Settings
          </Link>
        </DropdownMenuItem>
        <DropdownMenuItem asChild className="rounded-lg py-2.5 cursor-pointer">
          <Link href="/admin/profile">
            <UserCog className="mr-3 h-4 w-4" aria-hidden="true" />
            Manage Profile
          </Link>
        </DropdownMenuItem>
        <DropdownMenuSeparator className="my-2" />
        <DropdownMenuItem
          onClick={handleLogout}
          className="rounded-lg py-2.5 cursor-pointer text-destructive focus:text-destructive focus:bg-destructive/10"
        >
          <LogOut className="mr-3 h-4 w-4" aria-hidden="true" />
          Logout
        </DropdownMenuItem>
      </DropdownMenuContent>
    </DropdownMenu>
  )
}
