"use client"

import { useState, useMemo, useCallback } from "react"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { Button } from "@/components/ui/button"
import { Plus } from "lucide-react"
import { StateTable } from "./state-table"
import { AddStateDialog } from "./add-state-dialog"
import { EditStateDialog } from "./edit-state-dialog"
import { DeleteStateDialog } from "./delete-state-dialog"
import { locationDataStore, type State } from "@/lib/data/locations"
import { useToastEnhanced } from "@/hooks/use-toast-enhanced"

export function StatesClient() {
  const toast = useToastEnhanced()
  const [currentTab, setCurrentTab] = useState("all")

  const [states, setStates] = useState<State[]>(locationDataStore.getAllStates())
  const [isAddDialogOpen, setIsAddDialogOpen] = useState(false)
  const [editingState, setEditingState] = useState<State | null>(null)
  const [deletingState, setDeletingState] = useState<State | null>(null)

  const activeStates = useMemo(() => states.filter((s) => s.status === "active"), [states])
  const inactiveStates = useMemo(() => states.filter((s) => s.status === "inactive"), [states])

  const handleTabChange = (value: string) => {
    setCurrentTab(value)
  }

  const handleAddState = useCallback(
    (name: string) => {
      const newState = locationDataStore.addState(name)
      if (newState) {
        setStates(locationDataStore.getAllStates())
        toast.success({
          title: "State Added",
          description: `${name} has been added successfully.`,
        })
        setIsAddDialogOpen(false)
        return true
      }
      toast.error({
        title: "Failed to Add State",
        description: "A state with this name already exists.",
      })
      return false
    },
    [toast],
  )

  const handleEditState = useCallback(
    (id: string, name: string) => {
      const updated = locationDataStore.updateState(id, name)
      if (updated) {
        setStates(locationDataStore.getAllStates())
        toast.success({
          title: "State Updated",
          description: `${name} has been updated successfully.`,
        })
        setEditingState(null)
        return true
      }
      toast.error({
        title: "Failed to Update State",
        description: "A state with this name already exists.",
      })
      return false
    },
    [toast],
  )

  const handleToggleStatus = useCallback(
    (state: State) => {
      const updated = locationDataStore.toggleStateStatus(state.id)
      if (updated) {
        setStates(locationDataStore.getAllStates())
        const action = updated.status === "active" ? "activated" : "deactivated"
        toast.success({
          title: `State ${action.charAt(0).toUpperCase() + action.slice(1)}`,
          description: `${state.name} has been ${action}.`,
        })
      }
    },
    [toast],
  )

  const handleDeleteState = useCallback(() => {
    if (!deletingState) return

    const deleted = locationDataStore.deleteState(deletingState.id)
    if (deleted) {
      setStates(locationDataStore.getAllStates())
      toast.success({
        title: "State Deleted",
        description: `${deletingState.name} and all its cities have been deleted.`,
      })
    }
    setDeletingState(null)
  }, [deletingState, toast])

  const getDisplayStates = () => {
    switch (currentTab) {
      case "active":
        return activeStates
      case "inactive":
        return inactiveStates
      default:
        return states
    }
  }

  return (
    <section aria-labelledby="states-heading" className="admin-section">
      <header className="admin-page-header flex flex-col gap-4 sm:flex-row sm:items-center sm:justify-between">
        <div>
          <h1 id="states-heading" className="admin-page-title">
            States
          </h1>
          <p className="admin-page-description">Manage states for your platform coverage.</p>
        </div>
        <Button onClick={() => setIsAddDialogOpen(true)} className="w-full sm:w-auto">
          <Plus className="mr-2 h-4 w-4" aria-hidden="true" />
          Add State
        </Button>
      </header>

      <Tabs value={currentTab} onValueChange={handleTabChange} className="w-full">
        <TabsList className="mb-6 h-auto p-1.5 bg-muted/50 rounded-xl border border-border/50">
          <TabsTrigger
            value="all"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            All ({states.length})
          </TabsTrigger>
          <TabsTrigger
            value="active"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Active ({activeStates.length})
          </TabsTrigger>
          <TabsTrigger
            value="inactive"
            className="rounded-lg data-[state=active]:bg-background data-[state=active]:shadow-sm px-4 py-2"
          >
            Inactive ({inactiveStates.length})
          </TabsTrigger>
        </TabsList>

        <TabsContent value="all" className="mt-0">
          <StateTable
            states={getDisplayStates()}
            onEdit={setEditingState}
            onDelete={setDeletingState}
            onToggleStatus={handleToggleStatus}
          />
        </TabsContent>

        <TabsContent value="active" className="mt-0">
          <StateTable
            states={getDisplayStates()}
            onEdit={setEditingState}
            onDelete={setDeletingState}
            onToggleStatus={handleToggleStatus}
          />
        </TabsContent>

        <TabsContent value="inactive" className="mt-0">
          <StateTable
            states={getDisplayStates()}
            onEdit={setEditingState}
            onDelete={setDeletingState}
            onToggleStatus={handleToggleStatus}
          />
        </TabsContent>
      </Tabs>

      <AddStateDialog open={isAddDialogOpen} onOpenChange={setIsAddDialogOpen} onAdd={handleAddState} />

      <EditStateDialog
        state={editingState}
        open={!!editingState}
        onOpenChange={(open) => !open && setEditingState(null)}
        onSave={handleEditState}
      />

      <DeleteStateDialog
        state={deletingState}
        open={!!deletingState}
        onOpenChange={(open) => !open && setDeletingState(null)}
        onConfirm={handleDeleteState}
      />
    </section>
  )
}
