"use client"

import { useState, useEffect } from "react"
import { useForm } from "react-hook-form"
import { zodResolver } from "@hookform/resolvers/zod"
import { sanitizeInput } from "@/lib/security/sanitize"
import { Plus, Pencil, Trash2, User, Eye, EyeOff } from "lucide-react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
  DialogFooter,
  DialogClose,
} from "@/components/ui/dialog"
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from "@/components/ui/alert-dialog"
import { Badge } from "@/components/ui/badge"
import { useToastEnhanced } from "@/hooks/use-toast-enhanced"
import { createTeamMemberSchema, type CreateTeamMemberInput } from "@/lib/validations/roles"
import { rolesDataStore, type Role, type TeamMember } from "@/lib/data/roles"

export function ManageTeamClient() {
  const [teamMembers, setTeamMembers] = useState<TeamMember[]>([])
  const [roles, setRoles] = useState<Role[]>([])
  const [isDialogOpen, setIsDialogOpen] = useState(false)
  const [editingMember, setEditingMember] = useState<TeamMember | null>(null)
  const [showPassword, setShowPassword] = useState(false)
  const toast = useToastEnhanced()

  const {
    register,
    handleSubmit,
    reset,
    setValue,
    watch,
    formState: { errors, isSubmitting },
  } = useForm<CreateTeamMemberInput>({
    resolver: zodResolver(createTeamMemberSchema),
    defaultValues: {
      firstName: "",
      lastName: "",
      email: "",
      roleId: "",
      password: "",
    },
  })

  const selectedRoleId = watch("roleId")

  useEffect(() => {
    setTeamMembers(rolesDataStore.getAllTeamMembers())
    setRoles(rolesDataStore.getAllRoles())
  }, [])

  const getRoleName = (roleId: string): string => {
    const role = roles.find((r) => r.id === roleId)
    return role?.name ?? "Unknown Role"
  }

  const openCreateDialog = () => {
    setEditingMember(null)
    setShowPassword(false)
    reset({
      firstName: "",
      lastName: "",
      email: "",
      roleId: "",
      password: "",
    })
    setIsDialogOpen(true)
  }

  const openEditDialog = (member: TeamMember) => {
    setEditingMember(member)
    setShowPassword(false)
    reset({
      firstName: member.firstName,
      lastName: member.lastName,
      email: member.email,
      roleId: member.roleId,
      password: "",
    })
    setIsDialogOpen(true)
  }

  const onSubmit = async (data: CreateTeamMemberInput) => {
    const sanitizedData = {
      firstName: sanitizeInput(data.firstName),
      lastName: sanitizeInput(data.lastName),
      email: sanitizeInput(data.email),
      roleId: data.roleId,
      status: "active" as const,
    }

    const existingMember = rolesDataStore.getTeamMemberByEmail(sanitizedData.email)
    if (existingMember && (!editingMember || existingMember.id !== editingMember.id)) {
      toast.error({
        title: "Email already exists",
        description: "A team member with this email already exists.",
      })
      return
    }

    if (editingMember) {
      rolesDataStore.updateTeamMember(editingMember.id, sanitizedData)
      toast.success({
        title: "Team member updated",
        description: `${sanitizedData.firstName} ${sanitizedData.lastName} has been updated.`,
      })
    } else {
      rolesDataStore.createTeamMember(sanitizedData)
      toast.success({
        title: "Team member added",
        description: `${sanitizedData.firstName} ${sanitizedData.lastName} has been added to the team.`,
      })
    }

    setTeamMembers(rolesDataStore.getAllTeamMembers())
    setIsDialogOpen(false)
    reset()
    setEditingMember(null)
  }

  const handleDelete = (member: TeamMember) => {
    rolesDataStore.deleteTeamMember(member.id)
    setTeamMembers(rolesDataStore.getAllTeamMembers())
    toast.error({
      title: "Team member removed",
      description: `${member.firstName} ${member.lastName} has been removed.`,
    })
  }

  const toggleStatus = (member: TeamMember) => {
    const newStatus = member.status === "active" ? "inactive" : "active"
    rolesDataStore.updateTeamMember(member.id, { status: newStatus })
    setTeamMembers(rolesDataStore.getAllTeamMembers())
    toast.info({
      title: `Team member ${newStatus}`,
      description: `${member.firstName} ${member.lastName} is now ${newStatus}.`,
    })
  }

  return (
    <section aria-labelledby="team-section" className="space-y-6">
      <div className="flex justify-end">
        <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
          <DialogTrigger asChild>
            <Button onClick={openCreateDialog} className="rounded-xl shadow-lg shadow-primary/25">
              <Plus className="h-4 w-4 mr-2" aria-hidden="true" />
              Add Team Member
            </Button>
          </DialogTrigger>
          <DialogContent className="max-w-md">
            <DialogHeader>
              <DialogTitle>{editingMember ? "Edit Team Member" : "Add Team Member"}</DialogTitle>
              <DialogDescription>
                {editingMember ? "Update the team member details." : "Add a new team member and assign a role."}
              </DialogDescription>
            </DialogHeader>
            <form onSubmit={handleSubmit(onSubmit)} className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="firstName">First Name</Label>
                  <Input
                    id="firstName"
                    placeholder="John"
                    {...register("firstName")}
                    className="rounded-xl"
                    aria-describedby={errors.firstName ? "firstName-error" : undefined}
                  />
                  {errors.firstName && (
                    <p id="firstName-error" className="text-sm text-destructive">
                      {errors.firstName.message}
                    </p>
                  )}
                </div>

                <div className="space-y-2">
                  <Label htmlFor="lastName">Last Name</Label>
                  <Input
                    id="lastName"
                    placeholder="Doe"
                    {...register("lastName")}
                    className="rounded-xl"
                    aria-describedby={errors.lastName ? "lastName-error" : undefined}
                  />
                  {errors.lastName && (
                    <p id="lastName-error" className="text-sm text-destructive">
                      {errors.lastName.message}
                    </p>
                  )}
                </div>
              </div>

              <div className="space-y-2">
                <Label htmlFor="email">Email</Label>
                <Input
                  id="email"
                  type="email"
                  placeholder="john@example.com"
                  {...register("email")}
                  className="rounded-xl"
                  aria-describedby={errors.email ? "email-error" : undefined}
                />
                {errors.email && (
                  <p id="email-error" className="text-sm text-destructive">
                    {errors.email.message}
                  </p>
                )}
              </div>

              <div className="space-y-2">
                <Label htmlFor="roleId">Role</Label>
                <Select value={selectedRoleId} onValueChange={(value) => setValue("roleId", value)}>
                  <SelectTrigger id="roleId" className="rounded-xl">
                    <SelectValue placeholder="Select a role" />
                  </SelectTrigger>
                  <SelectContent>
                    {roles.map((role) => (
                      <SelectItem key={role.id} value={role.id}>
                        {role.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                {errors.roleId && <p className="text-sm text-destructive">{errors.roleId.message}</p>}
              </div>

              <div className="space-y-2">
                <Label htmlFor="password">Password {editingMember && "(leave blank to keep current)"}</Label>
                <div className="relative">
                  <Input
                    id="password"
                    type={showPassword ? "text" : "password"}
                    placeholder="••••••••"
                    {...register("password")}
                    className="rounded-xl"
                    aria-describedby={errors.password ? "password-error" : undefined}
                  />
                  <Button
                    type="button"
                    variant="ghost"
                    size="icon"
                    className="absolute right-0 top-0 h-full px-3 hover:bg-transparent"
                    onClick={() => setShowPassword(!showPassword)}
                    aria-label={showPassword ? "Hide password" : "Show password"}
                  >
                    {showPassword ? (
                      <EyeOff className="h-4 w-4" aria-hidden="true" />
                    ) : (
                      <Eye className="h-4 w-4" aria-hidden="true" />
                    )}
                  </Button>
                </div>
                {errors.password && (
                  <p id="password-error" className="text-sm text-destructive">
                    {errors.password.message}
                  </p>
                )}
              </div>

              <DialogFooter>
                <DialogClose asChild>
                  <Button type="button" variant="outline" className="rounded-xl bg-transparent">
                    Cancel
                  </Button>
                </DialogClose>
                <Button type="submit" disabled={isSubmitting} className="rounded-xl">
                  {isSubmitting ? "Saving..." : editingMember ? "Update Member" : "Add Member"}
                </Button>
              </DialogFooter>
            </form>
          </DialogContent>
        </Dialog>
      </div>

      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
        {teamMembers.map((member) => (
          <Card key={member.id} className="admin-card rounded-xl border-0">
            <CardHeader className="flex flex-row items-start justify-between space-y-0 pb-2">
              <div className="space-y-1">
                <CardTitle className="flex items-center gap-2 text-lg">
                  <div className="h-8 w-8 rounded-lg bg-gradient-to-br from-primary/20 to-primary/5 flex items-center justify-center">
                    <User className="h-4 w-4 text-primary" aria-hidden="true" />
                  </div>
                  {member.firstName} {member.lastName}
                </CardTitle>
                <CardDescription>{member.email}</CardDescription>
              </div>
              <div className="flex gap-1">
                <Button
                  variant="ghost"
                  size="icon"
                  onClick={() => openEditDialog(member)}
                  className="rounded-lg hover:bg-muted"
                  aria-label={`Edit ${member.firstName} ${member.lastName}`}
                >
                  <Pencil className="h-4 w-4" aria-hidden="true" />
                </Button>
                <AlertDialog>
                  <AlertDialogTrigger asChild>
                    <Button
                      variant="ghost"
                      size="icon"
                      className="rounded-lg hover:bg-destructive/10 hover:text-destructive"
                      aria-label={`Delete ${member.firstName} ${member.lastName}`}
                    >
                      <Trash2 className="h-4 w-4" aria-hidden="true" />
                    </Button>
                  </AlertDialogTrigger>
                  <AlertDialogContent>
                    <AlertDialogHeader>
                      <AlertDialogTitle>Remove Team Member</AlertDialogTitle>
                      <AlertDialogDescription>
                        Are you sure you want to remove "{member.firstName} {member.lastName}" from the team? This
                        action cannot be undone.
                      </AlertDialogDescription>
                    </AlertDialogHeader>
                    <AlertDialogFooter>
                      <AlertDialogCancel className="rounded-xl">Cancel</AlertDialogCancel>
                      <AlertDialogAction
                        onClick={() => handleDelete(member)}
                        className="bg-destructive text-destructive-foreground hover:bg-destructive/90 rounded-xl"
                      >
                        Remove
                      </AlertDialogAction>
                    </AlertDialogFooter>
                  </AlertDialogContent>
                </AlertDialog>
              </div>
            </CardHeader>
            <CardContent>
              <div className="flex items-center justify-between">
                <Badge variant="outline" className="rounded-lg">
                  {getRoleName(member.roleId)}
                </Badge>
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={() => toggleStatus(member)}
                  className={
                    member.status === "active"
                      ? "text-emerald-600 hover:text-emerald-700 hover:bg-emerald-500/10"
                      : "text-muted-foreground hover:bg-muted"
                  }
                >
                  {member.status === "active" ? "Active" : "Inactive"}
                </Button>
              </div>
            </CardContent>
          </Card>
        ))}
      </div>
    </section>
  )
}
