"use client"

import { useState, useMemo, useCallback } from "react"
import { DEFAULT_ITEMS_PER_PAGE } from "@/lib/constants/admin"
import type { SortDirection } from "@/lib/types/admin"

interface UseTableStateOptions<T, K extends keyof T> {
  data: T[]
  defaultSortField: K
  defaultSortDirection?: SortDirection
  defaultItemsPerPage?: number
  searchFields?: (keyof T)[]
}

export function useTableState<T extends Record<string, any>, K extends keyof T>({
  data,
  defaultSortField,
  defaultSortDirection = "asc",
  defaultItemsPerPage = DEFAULT_ITEMS_PER_PAGE,
  searchFields = [],
}: UseTableStateOptions<T, K>) {
  const [searchQuery, setSearchQuery] = useState("")
  const [currentPage, setCurrentPage] = useState(1)
  const [itemsPerPage, setItemsPerPage] = useState(defaultItemsPerPage)
  const [sortField, setSortField] = useState<K>(defaultSortField)
  const [sortDirection, setSortDirection] = useState<SortDirection>(defaultSortDirection)

  // Filter data based on search
  const filteredData = useMemo(() => {
    if (!searchQuery.trim() || searchFields.length === 0) return data

    const query = searchQuery.toLowerCase()
    return data.filter((item) =>
      searchFields.some((field) => {
        const value = item[field]
        return value && String(value).toLowerCase().includes(query)
      }),
    )
  }, [data, searchQuery, searchFields])

  // Sort data
  const sortedData = useMemo(() => {
    return [...filteredData].sort((a, b) => {
      const aValue = a[sortField]
      const bValue = b[sortField]

      let comparison = 0

      // Handle different data types
      if (typeof aValue === "string" && typeof bValue === "string") {
        comparison = aValue.localeCompare(bValue)
      } else if (typeof aValue === "number" && typeof bValue === "number") {
        comparison = aValue - bValue
      } else if (aValue instanceof Date && bValue instanceof Date) {
        comparison = aValue.getTime() - bValue.getTime()
      } else {
        // Fallback to string comparison
        comparison = String(aValue).localeCompare(String(bValue))
      }

      return sortDirection === "asc" ? comparison : -comparison
    })
  }, [filteredData, sortField, sortDirection])

  // Paginate data
  const paginatedData = useMemo(() => {
    const startIndex = (currentPage - 1) * itemsPerPage
    return sortedData.slice(startIndex, startIndex + itemsPerPage)
  }, [sortedData, currentPage, itemsPerPage])

  const totalPages = Math.ceil(sortedData.length / itemsPerPage)

  // Handle sort
  const handleSort = useCallback(
    (field: K) => {
      if (sortField === field) {
        setSortDirection((prev) => (prev === "asc" ? "desc" : "asc"))
      } else {
        setSortField(field)
        setSortDirection("asc")
      }
    },
    [sortField],
  )

  // Handle page change
  const handlePageChange = useCallback(
    (page: number) => {
      setCurrentPage(Math.max(1, Math.min(page, totalPages)))
    },
    [totalPages],
  )

  // Handle items per page change
  const handleItemsPerPageChange = useCallback((value: number) => {
    setItemsPerPage(value)
    setCurrentPage(1) // Reset to first page
  }, [])

  // Handle search change
  const handleSearchChange = useCallback((value: string) => {
    setSearchQuery(value)
    setCurrentPage(1) // Reset to first page on search
  }, [])

  // Reset all filters
  const resetFilters = useCallback(() => {
    setSearchQuery("")
    setCurrentPage(1)
    setItemsPerPage(defaultItemsPerPage)
    setSortField(defaultSortField)
    setSortDirection(defaultSortDirection)
  }, [defaultItemsPerPage, defaultSortField, defaultSortDirection])

  return {
    // Data
    paginatedData,
    filteredData,
    sortedData,
    totalItems: sortedData.length,
    originalDataLength: data.length,

    // Search
    searchQuery,
    setSearchQuery: handleSearchChange,

    // Pagination
    currentPage,
    itemsPerPage,
    totalPages,
    handlePageChange,
    handleItemsPerPageChange,

    // Sorting
    sortField,
    sortDirection,
    handleSort,

    // Utilities
    resetFilters,
    hasFilters: searchQuery.trim().length > 0,
  }
}
