export interface Permission {
  id: string
  name: string
  description: string
  category: string
}

export interface Role {
  id: string
  name: string
  description: string
  permissions: string[]
  createdAt: Date
  updatedAt: Date
}

export interface TeamMember {
  id: string
  firstName: string
  lastName: string
  email: string
  roleId: string
  status: "active" | "inactive"
  createdAt: Date
  updatedAt: Date
}

// Available permissions grouped by category
export const availablePermissions: Permission[] = [
  // Dashboard
  {
    id: "dashboard.view",
    name: "View Dashboard",
    description: "Access to view dashboard analytics",
    category: "Dashboard",
  },

  // User Management
  { id: "users.view", name: "View Users", description: "View user list and details", category: "User Management" },
  { id: "users.create", name: "Create Users", description: "Create new user accounts", category: "User Management" },
  { id: "users.edit", name: "Edit Users", description: "Edit user information", category: "User Management" },
  { id: "users.delete", name: "Delete Users", description: "Delete user accounts", category: "User Management" },

  // Content Management
  { id: "content.view", name: "View Content", description: "View content items", category: "Content Management" },
  { id: "content.create", name: "Create Content", description: "Create new content", category: "Content Management" },
  { id: "content.edit", name: "Edit Content", description: "Edit existing content", category: "Content Management" },
  { id: "content.delete", name: "Delete Content", description: "Delete content items", category: "Content Management" },
  {
    id: "content.publish",
    name: "Publish Content",
    description: "Publish content to live",
    category: "Content Management",
  },

  // Team Management
  { id: "team.view", name: "View Team", description: "View team members", category: "Team Management" },
  { id: "team.create", name: "Add Team Members", description: "Add new team members", category: "Team Management" },
  { id: "team.edit", name: "Edit Team Members", description: "Edit team member details", category: "Team Management" },
  { id: "team.delete", name: "Remove Team Members", description: "Remove team members", category: "Team Management" },

  // Role Management
  { id: "roles.view", name: "View Roles", description: "View available roles", category: "Role Management" },
  { id: "roles.create", name: "Create Roles", description: "Create new roles", category: "Role Management" },
  { id: "roles.edit", name: "Edit Roles", description: "Edit role permissions", category: "Role Management" },
  { id: "roles.delete", name: "Delete Roles", description: "Delete roles", category: "Role Management" },

  // Settings
  { id: "settings.view", name: "View Settings", description: "View system settings", category: "Settings" },
  { id: "settings.edit", name: "Edit Settings", description: "Modify system settings", category: "Settings" },
]

// Get permissions grouped by category
export function getPermissionsByCategory(): Record<string, Permission[]> {
  return availablePermissions.reduce(
    (acc, permission) => {
      if (!acc[permission.category]) {
        acc[permission.category] = []
      }
      acc[permission.category].push(permission)
      return acc
    },
    {} as Record<string, Permission[]>,
  )
}

// In-memory data store
class RolesDataStore {
  private roles: Map<string, Role> = new Map()
  private teamMembers: Map<string, TeamMember> = new Map()

  constructor() {
    // Initialize with default roles
    this.initializeDefaultRoles()
    this.initializeDefaultTeamMembers()
  }

  private initializeDefaultRoles() {
    const defaultRoles: Omit<Role, "id" | "createdAt" | "updatedAt">[] = [
      {
        name: "Super Admin",
        description: "Full access to all system features",
        permissions: availablePermissions.map((p) => p.id),
      },
      {
        name: "Content Manager",
        description: "Manage content and publications",
        permissions: [
          "dashboard.view",
          "content.view",
          "content.create",
          "content.edit",
          "content.delete",
          "content.publish",
        ],
      },
      {
        name: "User Manager",
        description: "Manage user accounts",
        permissions: ["dashboard.view", "users.view", "users.create", "users.edit"],
      },
      {
        name: "Viewer",
        description: "Read-only access to dashboard",
        permissions: ["dashboard.view"],
      },
    ]

    defaultRoles.forEach((role) => this.createRole(role))
  }

  private initializeDefaultTeamMembers() {
    const superAdminRole = Array.from(this.roles.values()).find((r) => r.name === "Super Admin")
    const contentManagerRole = Array.from(this.roles.values()).find((r) => r.name === "Content Manager")

    if (superAdminRole) {
      this.createTeamMember({
        firstName: "Admin",
        lastName: "User",
        email: "admin@teertham.com",
        roleId: superAdminRole.id,
        status: "active",
      })
    }

    if (contentManagerRole) {
      this.createTeamMember({
        firstName: "Content",
        lastName: "Manager",
        email: "content@teertham.com",
        roleId: contentManagerRole.id,
        status: "active",
      })
    }
  }

  // Role operations
  createRole(roleData: Omit<Role, "id" | "createdAt" | "updatedAt">): Role {
    const id = `role_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`
    const role: Role = {
      ...roleData,
      id,
      createdAt: new Date(),
      updatedAt: new Date(),
    }
    this.roles.set(id, role)
    return role
  }

  getRoleById(id: string): Role | undefined {
    return this.roles.get(id)
  }

  getAllRoles(): Role[] {
    return Array.from(this.roles.values())
  }

  updateRole(id: string, updates: Partial<Omit<Role, "id" | "createdAt">>): Role | undefined {
    const role = this.roles.get(id)
    if (!role) return undefined

    const updatedRole = {
      ...role,
      ...updates,
      updatedAt: new Date(),
    }
    this.roles.set(id, updatedRole)
    return updatedRole
  }

  deleteRole(id: string): boolean {
    return this.roles.delete(id)
  }

  // Team member operations
  createTeamMember(memberData: Omit<TeamMember, "id" | "createdAt" | "updatedAt">): TeamMember {
    const id = `member_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`
    const member: TeamMember = {
      ...memberData,
      id,
      createdAt: new Date(),
      updatedAt: new Date(),
    }
    this.teamMembers.set(id, member)
    return member
  }

  getTeamMemberById(id: string): TeamMember | undefined {
    return this.teamMembers.get(id)
  }

  getTeamMemberByEmail(email: string): TeamMember | undefined {
    return Array.from(this.teamMembers.values()).find((m) => m.email.toLowerCase() === email.toLowerCase())
  }

  getAllTeamMembers(): TeamMember[] {
    return Array.from(this.teamMembers.values())
  }

  updateTeamMember(id: string, updates: Partial<Omit<TeamMember, "id" | "createdAt">>): TeamMember | undefined {
    const member = this.teamMembers.get(id)
    if (!member) return undefined

    const updatedMember = {
      ...member,
      ...updates,
      updatedAt: new Date(),
    }
    this.teamMembers.set(id, updatedMember)
    return updatedMember
  }

  deleteTeamMember(id: string): boolean {
    return this.teamMembers.delete(id)
  }
}

// Export singleton instance
export const rolesDataStore = new RolesDataStore()
