import { spiritualPackageDataStore } from "./spiritual-packages"

export type SpiritualPackageBookingStatus = "booked" | "ongoing" | "completed" | "cancelled"

export interface SpiritualPackageBooking {
  id: string
  customerId: string
  customerName: string
  packageId: string
  packageCityName: string
  bookingDate: string
  travelersCount: number
  startDate: string
  endDate: string
  totalDays: number
  bookingAmount: number
  advanceAmount: number
  pendingAmount: number
  status: SpiritualPackageBookingStatus
  selectedVehicle: string
  selectedHotel: string
  createdAt: string
  updatedAt: string
}

class SpiritualPackageBookingDataStore {
  private bookings: Map<string, SpiritualPackageBooking> = new Map()
  private idCounter = 0

  constructor() {
    this.initializeSampleData()
  }

  private initializeSampleData() {
    const packages = spiritualPackageDataStore.getActivePackages()

    const customerNames = [
      "Ramesh Kumar Singh",
      "Sunita Devi Sharma",
      "Anil Kumar Patel",
      "Kavita Singh",
      "Suresh Yadav",
      "Meena Kumari",
      "Rajiv Gupta",
      "Anita Verma",
      "Vinod Mishra",
      "Geeta Rani",
      "Prakash Joshi",
      "Rekha Agarwal",
      "Mohan Lal Gupta",
      "Saroj Kumari",
      "Dinesh Choudhary",
      "Usha Devi",
      "Kishan Singh Rathore",
      "Poonam Sharma",
      "Ravi Prasad",
      "Seema Gupta",
    ]

    const sampleBookings: Array<{
      customerName: string
      packageIndex: number
      travelersCount: number
      bookingDate: string
      startDate: string
      status: SpiritualPackageBookingStatus
    }> = [
      // Booked (upcoming)
      {
        customerName: customerNames[0],
        packageIndex: 0,
        travelersCount: 4,
        bookingDate: "2024-02-01",
        startDate: "2024-03-15",
        status: "booked",
      },
      {
        customerName: customerNames[1],
        packageIndex: 1,
        travelersCount: 2,
        bookingDate: "2024-02-03",
        startDate: "2024-03-20",
        status: "booked",
      },
      {
        customerName: customerNames[2],
        packageIndex: 2,
        travelersCount: 6,
        bookingDate: "2024-02-05",
        startDate: "2024-03-25",
        status: "booked",
      },
      {
        customerName: customerNames[3],
        packageIndex: 0,
        travelersCount: 3,
        bookingDate: "2024-02-06",
        startDate: "2024-04-01",
        status: "booked",
      },
      {
        customerName: customerNames[4],
        packageIndex: 1,
        travelersCount: 5,
        bookingDate: "2024-02-08",
        startDate: "2024-04-05",
        status: "booked",
      },

      // Ongoing (in progress)
      {
        customerName: customerNames[5],
        packageIndex: 0,
        travelersCount: 2,
        bookingDate: "2024-01-20",
        startDate: "2024-02-08",
        status: "ongoing",
      },
      {
        customerName: customerNames[6],
        packageIndex: 1,
        travelersCount: 5,
        bookingDate: "2024-01-22",
        startDate: "2024-02-10",
        status: "ongoing",
      },
      {
        customerName: customerNames[7],
        packageIndex: 2,
        travelersCount: 4,
        bookingDate: "2024-01-25",
        startDate: "2024-02-11",
        status: "ongoing",
      },
      {
        customerName: customerNames[8],
        packageIndex: 0,
        travelersCount: 6,
        bookingDate: "2024-01-28",
        startDate: "2024-02-12",
        status: "ongoing",
      },

      // Completed (past)
      {
        customerName: customerNames[9],
        packageIndex: 0,
        travelersCount: 5,
        bookingDate: "2023-12-01",
        startDate: "2024-01-05",
        status: "completed",
      },
      {
        customerName: customerNames[10],
        packageIndex: 1,
        travelersCount: 2,
        bookingDate: "2023-12-05",
        startDate: "2024-01-10",
        status: "completed",
      },
      {
        customerName: customerNames[11],
        packageIndex: 2,
        travelersCount: 4,
        bookingDate: "2023-12-08",
        startDate: "2024-01-12",
        status: "completed",
      },
      {
        customerName: customerNames[12],
        packageIndex: 0,
        travelersCount: 3,
        bookingDate: "2023-12-10",
        startDate: "2024-01-15",
        status: "completed",
      },
      {
        customerName: customerNames[13],
        packageIndex: 1,
        travelersCount: 6,
        bookingDate: "2023-12-12",
        startDate: "2024-01-18",
        status: "completed",
      },
      {
        customerName: customerNames[14],
        packageIndex: 2,
        travelersCount: 7,
        bookingDate: "2023-12-18",
        startDate: "2024-01-22",
        status: "completed",
      },

      // Cancelled
      {
        customerName: customerNames[15],
        packageIndex: 0,
        travelersCount: 4,
        bookingDate: "2024-01-10",
        startDate: "2024-02-20",
        status: "cancelled",
      },
      {
        customerName: customerNames[16],
        packageIndex: 1,
        travelersCount: 3,
        bookingDate: "2024-01-12",
        startDate: "2024-02-25",
        status: "cancelled",
      },
      {
        customerName: customerNames[17],
        packageIndex: 2,
        travelersCount: 5,
        bookingDate: "2024-01-15",
        startDate: "2024-03-01",
        status: "cancelled",
      },
    ]

    sampleBookings.forEach((booking) => {
      const pkg = packages[booking.packageIndex % packages.length]

      if (pkg) {
        this.idCounter++
        const id = `spb_${String(this.idCounter).padStart(4, "0")}`
        const now = new Date().toISOString().split("T")[0]

        // Calculate total days and end date
        const totalDays = pkg.dayPlans.length
        const start = new Date(booking.startDate)
        const end = new Date(start)
        end.setDate(end.getDate() + totalDays - 1)
        const endDate = end.toISOString().split("T")[0]

        // Calculate amounts (using day 1 pricing as base)
        const basePrice = pkg.dayPlans[0]?.pricing.b2cWeekdayPerHead || 1500
        const totalAmount = basePrice * booking.travelersCount * totalDays
        const advanceAmount = Math.round(totalAmount * 0.3) // 30% advance
        const pendingAmount = booking.status === "cancelled" ? 0 : totalAmount - advanceAmount

        // Get vehicle and hotel names
        const selectedVehicle = pkg.dayPlans[0]?.vehicles[0]?.vehicleName || "Standard Vehicle"
        const selectedHotel = pkg.dayPlans[0]?.hotels[0]?.hotelName || "Standard Hotel"

        this.bookings.set(id, {
          id,
          customerId: `cust_${String(this.idCounter).padStart(3, "0")}`,
          customerName: booking.customerName,
          packageId: pkg.id,
          packageCityName: pkg.cityName,
          bookingDate: booking.bookingDate,
          travelersCount: booking.travelersCount,
          startDate: booking.startDate,
          endDate,
          totalDays,
          bookingAmount: booking.status === "cancelled" ? 0 : totalAmount,
          advanceAmount: booking.status === "cancelled" ? 0 : advanceAmount,
          pendingAmount,
          status: booking.status,
          selectedVehicle,
          selectedHotel,
          createdAt: now,
          updatedAt: now,
        })
      }
    })
  }

  getAllBookings(): SpiritualPackageBooking[] {
    return Array.from(this.bookings.values()).sort(
      (a, b) => new Date(b.startDate).getTime() - new Date(a.startDate).getTime(),
    )
  }

  getBookingsByStatus(status: SpiritualPackageBookingStatus): SpiritualPackageBooking[] {
    return this.getAllBookings().filter((b) => b.status === status)
  }

  getBookedBookings(): SpiritualPackageBooking[] {
    return this.getBookingsByStatus("booked")
  }

  getOngoingBookings(): SpiritualPackageBooking[] {
    return this.getBookingsByStatus("ongoing")
  }

  getCompletedBookings(): SpiritualPackageBooking[] {
    return this.getBookingsByStatus("completed")
  }

  getCancelledBookings(): SpiritualPackageBooking[] {
    return this.getBookingsByStatus("cancelled")
  }

  getBookingById(id: string): SpiritualPackageBooking | undefined {
    return this.bookings.get(id)
  }

  getBookingStats(): {
    booked: number
    ongoing: number
    completed: number
    cancelled: number
    total: number
    bookedAmount: number
    ongoingAmount: number
    completedAmount: number
    totalAmount: number
    totalTravelers: number
  } {
    const all = this.getAllBookings()
    const booked = all.filter((b) => b.status === "booked")
    const ongoing = all.filter((b) => b.status === "ongoing")
    const completed = all.filter((b) => b.status === "completed")
    const cancelled = all.filter((b) => b.status === "cancelled")

    return {
      booked: booked.length,
      ongoing: ongoing.length,
      completed: completed.length,
      cancelled: cancelled.length,
      total: all.length,
      bookedAmount: booked.reduce((sum, b) => sum + b.bookingAmount, 0),
      ongoingAmount: ongoing.reduce((sum, b) => sum + b.bookingAmount, 0),
      completedAmount: completed.reduce((sum, b) => sum + b.bookingAmount, 0),
      totalAmount: all.reduce((sum, b) => sum + b.bookingAmount, 0),
      totalTravelers: all.reduce((sum, b) => sum + b.travelersCount, 0),
    }
  }
}

export const spiritualPackageBookingDataStore = new SpiritualPackageBookingDataStore()
