export type UserRole = "customer" | "agent" | "admin" | "hotel"

export interface User {
  id: string
  name: string
  email: string
  phone: string
  password: string
  role: UserRole
  isVerified: boolean
  createdAt: Date
  updatedAt: Date
}

export interface OTPData {
  email: string
  otp: string
  expiresAt: Date
  purpose: "email-verification" | "password-reset"
}

// In-memory data store (replace with database in production)
class UserDataStore {
  private users: Map<string, User> = new Map()
  private otps: Map<string, OTPData> = new Map()
  private emailToUserId: Map<string, string> = new Map()
  private initialized = false

  // User operations
  createUser(userData: Omit<User, "id" | "createdAt" | "updatedAt">): User {
    const id = `user_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`
    const user: User = {
      ...userData,
      id,
      createdAt: new Date(),
      updatedAt: new Date(),
    }
    this.users.set(id, user)
    this.emailToUserId.set(user.email.toLowerCase(), id)
    return user
  }

  getUserById(id: string): User | undefined {
    return this.users.get(id)
  }

  getUserByEmail(email: string): User | undefined {
    const userId = this.emailToUserId.get(email.toLowerCase())
    return userId ? this.users.get(userId) : undefined
  }

  updateUser(id: string, updates: Partial<User>): User | undefined {
    const user = this.users.get(id)
    if (!user) return undefined

    const updatedUser = {
      ...user,
      ...updates,
      updatedAt: new Date(),
    }
    this.users.set(id, updatedUser)
    return updatedUser
  }

  deleteUser(id: string): boolean {
    const user = this.users.get(id)
    if (!user) return false

    this.emailToUserId.delete(user.email.toLowerCase())
    return this.users.delete(id)
  }

  // OTP operations
  createOTP(email: string, purpose: OTPData["purpose"]): string {
    const otp = Math.floor(100000 + Math.random() * 900000).toString()
    const expiresAt = new Date(Date.now() + 10 * 60 * 1000) // 10 minutes

    this.otps.set(email.toLowerCase(), {
      email: email.toLowerCase(),
      otp,
      expiresAt,
      purpose,
    })

    return otp
  }

  verifyOTP(email: string, otp: string, purpose: OTPData["purpose"]): boolean {
    const otpData = this.otps.get(email.toLowerCase())
    if (!otpData) return false

    if (otpData.purpose !== purpose) return false
    if (otpData.expiresAt < new Date()) {
      this.otps.delete(email.toLowerCase())
      return false
    }

    const isValid = otpData.otp === otp
    if (isValid) {
      this.otps.delete(email.toLowerCase())
    }

    return isValid
  }

  deleteOTP(email: string): boolean {
    return this.otps.delete(email.toLowerCase())
  }

  getOTP(email: string, purpose: OTPData["purpose"]): { code: string; expiresAt: Date } | null {
    const otpData = this.otps.get(email.toLowerCase())
    if (!otpData) return null
    if (otpData.purpose !== purpose) return null
    if (otpData.expiresAt < new Date()) {
      this.otps.delete(email.toLowerCase())
      return null
    }
    return { code: otpData.otp, expiresAt: otpData.expiresAt }
  }

  initializeDefaults(): void {
    if (this.initialized) return
    this.initialized = true

    // Admin account
    if (!this.getUserByEmail(defaultCredentials.admin.email)) {
      this.createUser({
        name: defaultCredentials.admin.name,
        email: defaultCredentials.admin.email,
        phone: "",
        password: defaultCredentials.admin.password,
        role: defaultCredentials.admin.role,
        isVerified: true,
      })
    }

    // Customer account
    if (!this.getUserByEmail(defaultCredentials.customer.email)) {
      this.createUser({
        name: defaultCredentials.customer.name,
        email: defaultCredentials.customer.email,
        phone: defaultCredentials.customer.phone,
        password: defaultCredentials.customer.password,
        role: defaultCredentials.customer.role,
        isVerified: true,
      })
    }

    // Agent account
    if (!this.getUserByEmail(defaultCredentials.agent.email)) {
      this.createUser({
        name: defaultCredentials.agent.name,
        email: defaultCredentials.agent.email,
        phone: defaultCredentials.agent.phone,
        password: defaultCredentials.agent.password,
        role: defaultCredentials.agent.role,
        isVerified: true,
      })
    }

    // Hotel account
    if (!this.getUserByEmail(defaultCredentials.hotel.email)) {
      this.createUser({
        name: defaultCredentials.hotel.name,
        email: defaultCredentials.hotel.email,
        phone: defaultCredentials.hotel.phone,
        password: defaultCredentials.hotel.password,
        role: defaultCredentials.hotel.role,
        isVerified: true,
      })
    }
  }
}

// Export singleton instance
export const userDataStore = new UserDataStore()

export const defaultCredentials = {
  admin: {
    email: "admin@teertham.com",
    password: "Admin@123",
    name: "Admin User",
    role: "admin" as const,
  },
  customer: {
    email: "customer@teertham.com",
    password: "Customer@123",
    name: "Test Customer",
    phone: "+91 9876543210",
    role: "customer" as const,
  },
  agent: {
    email: "agent@teertham.com",
    password: "Agent@123",
    name: "Test Agent",
    phone: "+91 9876543211",
    role: "agent" as const,
  },
  hotel: {
    email: "hotel@teertham.com",
    password: "Hotel@123",
    name: "Test Hotel",
    phone: "+91 9876543212",
    role: "hotel" as const,
  },
}

export const defaultAccounts = defaultCredentials

export const TEST_OTP = "123456"

// Only initialize on client-side and only once
if (typeof window !== "undefined") {
  userDataStore.initializeDefaults()
}
